<?php
session_start();
require_once 'config/config_bd.php';
require_once 'lib/usuario_utils.php';
requireAuth();

// CRIAR NOVA RESERVA
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['form_tipo'] ?? '') === 'nova_reserva') {
    $usuario_id = (int) $_POST['usuario_id'];
    $livro_id = (int) $_POST['livro_id'];

    // Bloquear se tiver débito
    if (usuario_tem_debito($conexao, $usuario_id)) {
        $_SESSION['mensagem_erro'] = "Usuário possui débito pendente e não pode fazer reservas até quitar a multa.";
        header('Location: listar_reservas.php');
        exit();
    }

    // Inserir reserva (não altera quantidade_disponivel)
    $stmt = $conexao->prepare("INSERT INTO reservas (usuario_id, livro_id) VALUES (?, ?)");
    $stmt->bind_param("ii", $usuario_id, $livro_id);
    if ($stmt->execute()) {
        $_SESSION['mensagem_sucesso'] = "Reserva criada com sucesso.";
    } else {
        $_SESSION['mensagem_erro'] = "Erro ao criar reserva: " . $stmt->error;
    }
    $stmt->close();
    header('Location: listar_reservas.php');
    exit();
}

// CANCELAR RESERVA
if (isset($_GET['cancelar'])) {
    $id = (int) $_GET['cancelar'];
    $stmt = $conexao->prepare("UPDATE reservas SET status = 'cancelada' WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        $_SESSION['mensagem_sucesso'] = "Reserva cancelada.";
    } else {
        $_SESSION['mensagem_erro'] = "Erro ao cancelar reserva.";
    }
    $stmt->close();
    header('Location: listar_reservas.php');
    exit();
}

// CONVERTER RESERVA EM EMPRÉSTIMO
if (isset($_GET['converter'])) {
    $res_id = (int) $_GET['converter'];

    // buscar reserva
    $stmt = $conexao->prepare("SELECT usuario_id, livro_id, status FROM reservas WHERE id = ?");
    $stmt->bind_param("i", $res_id);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$res || $res['status'] !== 'ativa') {
        $_SESSION['mensagem_erro'] = "Reserva inválida para conversão.";
        header('Location: listar_reservas.php');
        exit();
    }

    $usuario_id = (int) $res['usuario_id'];
    $livro_id = (int) $res['livro_id'];

    // verificar disponibilidade
    $v = $conexao->prepare("SELECT quantidade_disponivel FROM livros WHERE id = ?");
    $v->bind_param("i", $livro_id);
    $v->execute();
    $row = $v->get_result()->fetch_assoc();
    $v->close();

    if (!$row || $row['quantidade_disponivel'] <= 0) {
        $_SESSION['mensagem_erro'] = "Livro sem exemplares disponíveis para empréstimo.";
        header('Location: listar_reservas.php');
        exit();
    }

    // criar empréstimo (data_devolucao_prevista pedimos 7 dias por padrão)
    $data_prevista = date('Y-m-d', strtotime('+7 days'));
    $stmt = $conexao->prepare("INSERT INTO emprestimos (usuario_id, livro_id, data_devolucao_prevista) VALUES (?, ?, ?)");
    $stmt->bind_param("iis", $usuario_id, $livro_id, $data_prevista);
    if ($stmt->execute()) {
        // atualizar quantidade_disponivel e status do livro
        $u = $conexao->prepare("UPDATE livros SET quantidade_disponivel = quantidade_disponivel - 1,
                               status = CASE WHEN quantidade_disponivel - 1 <= 0 THEN 'emprestado' ELSE 'ativo' END
                               WHERE id = ?");
        $u->bind_param("i", $livro_id);
        $u->execute();
        $u->close();

        // marcar reserva como convertida
        $m = $conexao->prepare("UPDATE reservas SET status = 'convertida' WHERE id = ?");
        $m->bind_param("i", $res_id);
        $m->execute();
        $m->close();

        $_SESSION['mensagem_sucesso'] = "Reserva convertida em empréstimo.";
    } else {
        $_SESSION['mensagem_erro'] = "Erro ao criar empréstimo: " . $stmt->error;
    }
    $stmt->close();
    header('Location: listar_reservas.php');
    exit();
}

// buscar reservas (todas)
$sql = "SELECT r.*, u.nome AS usuario_nome, l.titulo AS livro_titulo
        FROM reservas r
        JOIN usuarios u ON r.usuario_id = u.id
        JOIN livros l ON r.livro_id = l.id
        ORDER BY r.data_reserva DESC";
$reservas = mysqli_query($conexao, $sql);
if (!$reservas) {
    die("Erro na query de reservas: " . mysqli_error($conexao));
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>Reservas - Biblioteca</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body>
<div class="container mt-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="display-5">Reservas</h1>
        <div>
            <?php if (usuario_logado()): ?>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addModal">
                <i class="bi bi-calendar-plus me-2"></i>Nova Reserva
            </button>
            <?php endif; ?>
            <a href="index.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left me-2"></i>Voltar
            </a>
        </div>
    </div>

    <?php if (isset($_SESSION['mensagem_sucesso'])): ?>
        <div class="alert alert-success"><?= $_SESSION['mensagem_sucesso']; ?></div>
        <?php unset($_SESSION['mensagem_sucesso']); endif; ?>
    <?php if (isset($_SESSION['mensagem_erro'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['mensagem_erro']; ?></div>
        <?php unset($_SESSION['mensagem_erro']); endif; ?>

    <div class="card shadow">
        <table class="table table-hover mb-0">
            <thead class="table-light">
                <tr>
                    <th>ID</th>
                    <th>Usuário</th>
                    <th>Livro</th>
                    <th>Data Reserva</th>
                    <th>Status</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody>
            <?php if (mysqli_num_rows($reservas) > 0): while ($r = mysqli_fetch_assoc($reservas)): ?>
                <tr>
                    <td><?= $r['id'] ?></td>
                    <td><?= htmlspecialchars($r['usuario_nome']) ?></td>
                    <td><?= htmlspecialchars($r['livro_titulo']) ?></td>
                    <td><?= date('d/m/Y H:i', strtotime($r['data_reserva'])) ?></td>
                    <td>
                        <?php if ($r['status'] === 'ativa'): ?>
                            <span class="badge bg-primary">Ativa</span>
                        <?php elseif ($r['status'] === 'cancelada'): ?>
                            <span class="badge bg-secondary">Cancelada</span>
                        <?php else: ?>
                            <span class="badge bg-success">Convertida</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($r['status'] === 'ativa'): ?>
                            <a href="?converter=<?= $r['id'] ?>" class="btn btn-sm btn-success"
                               onclick="return confirm('Converter reserva em empréstimo?')">
                                <i class="bi bi-arrow-right-circle"></i> Converter
                            </a>
                            <a href="?cancelar=<?= $r['id'] ?>" class="btn btn-sm btn-danger"
                               onclick="return confirm('Cancelar reserva?')">
                                <i class="bi bi-x-circle"></i> Cancelar
                            </a>
                        <?php else: ?>
                            <small class="text-muted">Sem ações</small>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endwhile; else: ?>
                <tr><td colspan="6" class="text-center py-3">Nenhuma reserva encontrada.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal: Nova Reserva -->
<div class="modal fade" id="addModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <input type="hidden" name="form_tipo" value="nova_reserva">
                <div class="modal-header">
                    <h5 class="modal-title">Nova Reserva</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Usuário</label>
                        <select name="usuario_id" class="form-control" required>
                            <option value="">Selecione...</option>
                            <?php
                            $us = mysqli_query($conexao, "
    SELECT id, nome 
    FROM usuarios 
    WHERE id NOT IN (SELECT usuario_id FROM emprestimos WHERE multa > 0 AND multa_paga = 0)
    ORDER BY nome
");
while ($u = mysqli_fetch_assoc($us)) {
    echo "<option value='{$u['id']}'>" . htmlspecialchars($u['nome']) . "</option>";
}
                            ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Livro</label>
                        <select name="livro_id" class="form-control" required>
                            <option value="">Selecione...</option>
                            <?php
                            $ls = mysqli_query($conexao, "SELECT id, titulo, quantidade_disponivel FROM livros ORDER BY titulo");
                            while ($l = mysqli_fetch_assoc($ls)) {
                                $suf = $l['quantidade_disponivel'] <= 0 ? " (sem exemplares)" : "";
                                echo "<option value='{$l['id']}'>" . htmlspecialchars($l['titulo']) . $suf . "</option>";
                            }
                            ?>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">Cancelar</button>
                    <button class="btn btn-primary" type="submit">Reservar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
<?php mysqli_close($conexao); ?>