<?php
session_start();
require_once 'config/config_bd.php';
require_once 'lib/usuario_utils.php';
requireAuth();

// REGISTRAR NOVO EMPRÉSTIMO
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['form_tipo'] ?? '') === 'novo_emprestimo') {
    $usuario_id = (int) $_POST['usuario_id'];
    $livro_id = (int) $_POST['livro_id'];
    $data_devolucao_prevista = $_POST['data_devolucao_prevista'];

    // Verificar bloqueio por débito
    if (usuario_tem_debito($conexao, $usuario_id)) {
        $_SESSION['mensagem_erro'] = "Usuário possui débito pendente e não pode pegar empréstimos até quitar a multa.";
        header('Location: listar_emprestimos.php');
        exit();
    }

    // Verificar se livro está disponível
    $verificar = $conexao->prepare("SELECT quantidade_disponivel FROM livros WHERE id = ?");
    $verificar->bind_param("i", $livro_id);
    $verificar->execute();
    $resultado = $verificar->get_result()->fetch_assoc();

    if ($resultado['quantidade_disponivel'] <= 0) {
        $_SESSION['mensagem_erro'] = "Livro não disponível para empréstimo!";
    } else {
        // Registrar empréstimo
        $stmt = $conexao->prepare("INSERT INTO emprestimos (usuario_id, livro_id, data_devolucao_prevista) 
                                  VALUES (?, ?, ?)");
        $stmt->bind_param("iis", $usuario_id, $livro_id, $data_devolucao_prevista);

        if ($stmt->execute()) {
            // Atualizar quantidade disponível e status do livro
            $update = $conexao->prepare("UPDATE livros SET quantidade_disponivel = quantidade_disponivel - 1,
                                        status = CASE WHEN quantidade_disponivel - 1 <= 0 THEN 'emprestado' ELSE 'ativo' END
                                        WHERE id = ?");
            $update->bind_param("i", $livro_id);
            $update->execute();
            $update->close();

            $_SESSION['mensagem_sucesso'] = "Empréstimo registrado com sucesso!";
        } else {
            $_SESSION['mensagem_erro'] = "Erro ao registrar: " . $stmt->error;
        }
        $stmt->close();
    }
    $verificar->close();

    header('Location: listar_emprestimos.php');
    exit();
}

// REGISTRAR DEVOLUÇÃO
if (isset($_GET['devolver'])) {
    $emprestimo_id = (int) $_GET['devolver'];

    // Obter dados do empréstimo
    $buscar = $conexao->prepare("SELECT livro_id FROM emprestimos WHERE id = ?");
    $buscar->bind_param("i", $emprestimo_id);
    $buscar->execute();
    $emp = $buscar->get_result()->fetch_assoc();
    $livro_id = $emp['livro_id'];
    $buscar->close();

    // Atualizar empréstimo
    $stmt = $conexao->prepare("UPDATE emprestimos SET status = 'devolvido', data_devolucao_real = NOW() WHERE id = ?");
    $stmt->bind_param("i", $emprestimo_id);

    if ($stmt->execute()) {
        // Aumentar quantidade disponível
        $update = $conexao->prepare("UPDATE livros SET quantidade_disponivel = quantidade_disponivel + 1,
                                    status = 'ativo' WHERE id = ?");
        $update->bind_param("i", $livro_id);
        $update->execute();
        $update->close();

        $_SESSION['mensagem_sucesso'] = "Livro devolvido com sucesso!";
    } else {
        $_SESSION['mensagem_erro'] = "Erro ao registrar devolução!";
    }
    $stmt->close();

    header('Location: listar_emprestimos.php');
    exit();
}

// Buscar empréstimos ativos
$sql = "SELECT e.*, u.nome as usuario_nome, l.titulo as livro_titulo
        FROM emprestimos e
        JOIN usuarios u ON e.usuario_id = u.id
        JOIN livros l ON e.livro_id = l.id
        WHERE e.status = 'ativo'
        ORDER BY e.data_devolucao_prevista ASC";
$emprestimos = mysqli_query($conexao, $sql);

if (!$emprestimos) {
    die("Erro na query: " . mysqli_error($conexao));
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>Empréstimos - Biblioteca</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body>

<div class="container mt-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="display-5">Empréstimos da Biblioteca</h1>
        <div>
            <?php if (isAdmin()): ?>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addModal">
                <i class="bi bi-plus-circle me-2"></i>Novo Empréstimo
            </button>
            <?php endif; ?>
            <a href="index.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left me-2"></i>Voltar
            </a>
        </div>
    </div>

    <!-- Mensagens -->
    <?php if (isset($_SESSION['mensagem_sucesso'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= $_SESSION['mensagem_sucesso'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['mensagem_sucesso']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['mensagem_erro'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= $_SESSION['mensagem_erro'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['mensagem_erro']); ?>
    <?php endif; ?>

    <!-- Tabela de Empréstimos -->
    <div class="card shadow">
        <table class="table table-hover mb-0">
            <thead class="table-light">
                <tr>
                    <th>ID</th>
                    <th>Usuário</th>
                    <th>Livro</th>
                    <th>Data Empréstimo</th>
                    <th>Data Devolução</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody>
                <?php
                if (mysqli_num_rows($emprestimos) > 0) {
                    while ($emp = mysqli_fetch_assoc($emprestimos)) {
                        $data_dev = new DateTime($emp['data_devolucao_prevista']);
                        $hoje = new DateTime();
                        $atrasado = $data_dev < $hoje ? 'table-danger' : '';
                        ?>
                        <tr class="<?= $atrasado ?>">
                            <td><?= $emp['id'] ?></td>
                            <td><?= htmlspecialchars($emp['usuario_nome']) ?></td>
                            <td><?= htmlspecialchars($emp['livro_titulo']) ?></td>
                            <td><?= date('d/m/Y', strtotime($emp['data_emprestimo'])) ?></td>
                            <td><?= date('d/m/Y', strtotime($emp['data_devolucao_prevista'])) ?></td>
                            <td>
                                <?php if (isAdmin()): ?>
                                <a href="?devolver=<?= $emp['id'] ?>" class="btn btn-success btn-sm"
                                   onclick="return confirm('Confirmar devolução?')">
                                    <i class="bi bi-check-circle"></i> Devolver
                                </a>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php
                    }
                } else {
                    echo "<tr><td colspan='6' class='text-center py-3'>Nenhum empréstimo ativo.</td></tr>";
                }
                ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal: Novo Empréstimo -->
<?php if (isAdmin()): ?>
<div class="modal fade" id="addModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Registrar Novo Empréstimo</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="form_tipo" value="novo_emprestimo">

                    <div class="mb-3">
                        <label class="form-label">Usuário:</label>
                        <select name="usuario_id" class="form-control" required>
                            <option value="">Selecione um usuário</option>
                            <?php
                            $usuarios = mysqli_query($conexao, "SELECT id, nome FROM usuarios ORDER BY nome");
                            while ($u = mysqli_fetch_assoc($usuarios)) {
                                echo "<option value='{$u['id']}'>" . htmlspecialchars($u['nome']) . "</option>";
                            }
                            ?>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Livro:</label>
                        <select name="livro_id" class="form-control" required>
                            <option value="">Selecione um livro</option>
                            <?php
                            $livros = mysqli_query($conexao, "SELECT id, titulo FROM livros WHERE quantidade_disponivel > 0 ORDER BY titulo");
                            while ($l = mysqli_fetch_assoc($livros)) {
                                echo "<option value='{$l['id']}'>" . htmlspecialchars($l['titulo']) . "</option>";
                            }
                            ?>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Data de Devolução Prevista:</label>
                        <input type="date" name="data_devolucao_prevista" class="form-control" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Registrar Empréstimo</button>
                </div>
            </form>
        </div>
    </div>
</div>
<?php endif; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
<?php mysqli_close($conexao); ?>